﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;
    using Hims.Shared.Library.Enums;
    using Hims.Api.Models;

    /// <inheritdoc />
    [Authorize]
    [Route("api/icd-code")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class IcdCodeController : BaseController
    {
        /// <summary>
        /// The icdCode services.
        /// </summary>
        private readonly IIcdService icdCodeService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public IcdCodeController(IIcdService icdCodeService, IAuditLogService auditLogServices) { this.icdCodeService = icdCodeService; this.auditLogServices = auditLogServices; }

        /// <summary>
        /// The fetch ICDCodes.
        /// </summary>
        /// <param name="model">
        /// The ICDCode filter model.
        /// </param>
        /// <returns>
        /// The list of ICDCodes.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of ICDCodes.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ICDCodeModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]IcdCodeFilterModel model)
        {
            model = (IcdCodeFilterModel)EmptyFilter.Handler(model);
            var icdCodes = await this.icdCodeService.FetchAsync(model);
            return this.Success(icdCodes);
        }

        /// <summary>
        /// The add icd codes.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - ICDCode added successfully.
        /// - 409 - ICDCode already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody]ICDCodeModel model, [FromHeader] LocationHeader header)
        {
            model = (ICDCodeModel)EmptyFilter.Handler(model);
            var response = await this.icdCodeService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given ICDCode name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.ICDCodes,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Code} ICDCode has been added.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success("ICDCode has been added successfully.");
        }

        /// <summary>
        /// The update ICDCode.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - ICDCode updated successfully.
        /// - 409 - ICDCode already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody]ICDCodeModel model, [FromHeader] LocationHeader header)
        {
            model = (ICDCodeModel)EmptyFilter.Handler(model);
            var response = await this.icdCodeService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given ICDCode name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.ICDCodes,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Code} ICDCode has been updated successfully.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success("ICDCode has been updated successfully.");
        }

        /// <summary>
        /// The delete ICDCode.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - ICDCode deleted successfully.
        /// - 409 - ICDCode can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody]ICDCodeModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (ICDCodeModel)EmptyFilter.Handler(model);
                var response = await this.icdCodeService.DeleteAsync(model.ICDCodeId);
                var responseProviderEncounter = await this.icdCodeService.updateProviderEncounterAsync(model.ICDCodeId);
                if (response == 0)
                {
                    return this.ServerError();
                }
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.ICDCodes,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"{model.Code} ICDCode has been deleted successfully.",
                    LocationId= Convert.ToInt32(header.LocationId)

                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success("ICDCode has been deleted successfully.");
            }
            catch (NpgsqlException)
            {
                return this.ServerError();
            }
        }
    }
}